with
  text_io,
  ada.integer_text_io,
  ada.long_float_text_io,
  ada.numerics.long_elementary_functions;

procedure icosahedron is

  edge     : array (integer range 1 .. 30, integer range 1 .. 2) of integer;
  xo,yo,zo : array (integer range 1 .. 12) of long_float;

  procedure initia is

    use
      ada.numerics.long_elementary_functions;

    c1,c2 : long_float;

  begin
    -- vertices coordinates
    c1 := sqrt(2.0 / (sqrt(5.0) + 5.0));
    c2 := sqrt((sqrt(5.0) + 5.0) / 10.0);

    xo(1) := c1;
    yo(1) := 0.0;
    zo(1) := c2;

    xo(2) := -c1;
    yo(2) := 0.0;
    zo(2) := c2;

    xo(3) := 0.0;
    yo(3) := c2;
    zo(3) := c1;

    xo(4) := 0.0;
    yo(4) := -c2;
    zo(4) := c1;

    xo(5) := c2;
    yo(5) := c1;
    zo(5) := 0.0;

    xo(6) := -c2;
    yo(6) := c1;
    zo(6) := 0.0;

    xo(7) := -c2;
    yo(7) := -c1;
    zo(7) := 0.0;

    xo(8) := c2;
    yo(8) := -c1;
    zo(8) := 0.0;

    xo(9) := 0.0;
    yo(9) := c2;
    zo(9) := -c1;

    xo(10) := 0.0;
    yo(10) := -c2;
    zo(10) := -c1;

    xo(11) := c1;
    yo(11) := 0.0;
    zo(11) := -c2;

    xo(12) := -c1;
    yo(12) := 0.0;
    zo(12) := -c2;

    -- edges
    edge(1,1) := 1;
    edge(1,2) := 2;

    edge(2,1) := 1;
    edge(2,2) := 3;

    edge(3,1) := 1;
    edge(3,2) := 4;

    edge(4,1) := 1;
    edge(4,2) := 5;

    edge(5,1) := 1;
    edge(5,2) := 8;

    edge(6,1) := 2;
    edge(6,2) := 3;

    edge(7,1) := 2;
    edge(7,2) := 4;

    edge(8,1) := 2;
    edge(8,2) := 6;

    edge(9,1) := 2;
    edge(9,2) := 7;

    edge(10,1) := 3;
    edge(10,2) := 5;

    edge(11,1) := 3;
    edge(11,2) := 6;

    edge(12,1) := 3;
    edge(12,2) := 9;

    edge(13,1) := 4;
    edge(13,2) := 7;

    edge(14,1) := 4;
    edge(14,2) := 8;

    edge(15,1) := 4;
    edge(15,2) := 10;

    edge(16,1) := 5;
    edge(16,2) := 8;

    edge(17,1) := 5;
    edge(17,2) := 9;

    edge(18,1) := 5;
    edge(18,2) := 11;

    edge(19,1) := 6;
    edge(19,2) := 7;

    edge(20,1) := 6;
    edge(20,2) := 9;

    edge(21,1) := 6;
    edge(21,2) := 12;

    edge(22,1) := 7;
    edge(22,2) := 10;

    edge(23,1) := 7;
    edge(23,2) := 12;

    edge(24,1) := 8;
    edge(24,2) := 10;

    edge(25,1) := 8;
    edge(25,2) := 11;

    edge(26,1) := 9;
    edge(26,2) := 11;

    edge(27,1) := 9;
    edge(27,2) := 12;

    edge(28,1) := 10;
    edge(28,2) := 11;

    edge(29,1) := 10;
    edge(29,2) := 12;

    edge(30,1) := 11;
    edge(30,2) := 12;
  end initia;

  procedure input is

    use
      text_io,
      ada.integer_text_io,
      ada.long_float_text_io;

    maxls,maxminor,maxlist,h    : integer;
    lstol,mntol,mnref,mu10,mu01 : long_float;
    io                          : file_type;

  begin
    create(io, out_file, "input.txt");

    -- solution parameters
    lstol := 1.0E-01;
    maxls := 20;
    mntol := 1.0E-05;
    mnref := 1.0;
    maxminor := 360;
    maxlist := 3;

    put(io,"        lstol     maxls             mntol             mnref  maxminor   maxlist");
    new_line(io);
    put(io,":");
    new_line(io);
    put(io,lstol,2,6,3);
    put(io,"     ");
    put(io,maxls,5);
    put(io,"     ");
    put(io,mntol,2,6,3);
    put(io,"     ");
    put(io,mnref,2,6,3);
    put(io,"     ");
    put(io,maxminor,5);
    put(io,"     ");
    put(io,maxlist,5);
    new_line(io);

    -- material parameters
    h := 2;
    mu10 := 0.3750;
    mu01 := -0.1250;

    new_line(io);
    new_line(io);
    put(io,"    h              mu10              mu01");
    new_line(io);
    put(io,":");
    new_line(io);
    put(io,h,5);
    put(io,"     ");
    put(io,mu10,2,6,3);
    put(io,"     ");
    put(io,mu01,2,6,3);
    new_line(io);

    -- node coordinates
    new_line(io);
    new_line(io);
    put(io," node           coord x           coord y           coord z");
    new_line(io);
    put(io,":");
    new_line(io);
    put(io,"   12");
    new_line(io);

    new_line(io);
    for glob in 1 .. 12 loop
      put(io,glob,5);
      put(io,"     ");
      put(io,xo(glob),2,6,3);
      put(io,"     ");
      put(io,yo(glob),2,6,3);
      put(io,"     ");
      put(io,zo(glob),2,6,3);
      new_line(io);
    end loop;

    -- elements
    new_line(io);
    new_line(io);
    put(io," elem      node      node              area");
    new_line(io);
    put(io,":");
    new_line(io);
    put(io,"   30");
    new_line(io);

    new_line(io);
    for e in 1 .. 30 loop
      put(io,e,5);
      for loc in 1 .. 2 loop
        put(io,"     ");
        put(io,edge(e,loc),5);
      end loop;
      put(io,"      1.000000E+00");
      new_line(io);
    end loop;

    -- known displacements
    new_line(io);
    new_line(io);
    put(io," node      axis             displ");
    new_line(io);
    put(io,":");
    new_line(io);
    put(io,"   12");
    new_line(io);

    new_line(io);

    put(io,"    3         1      0.000000E+00");
    new_line(io);
    put(io,"    4         1      0.000000E+00");
    new_line(io);
    put(io,"    9         1      0.000000E+00");
    new_line(io);
    put(io,"   10         1      0.000000E+00");
    new_line(io);

    put(io,"    1         2      0.000000E+00");
    new_line(io);
    put(io,"    2         2      0.000000E+00");
    new_line(io);
    put(io,"   11         2      0.000000E+00");
    new_line(io);
    put(io,"   12         2      0.000000E+00");
    new_line(io);

    put(io,"    5         3      0.000000E+00");
    new_line(io);
    put(io,"    6         3      0.000000E+00");
    new_line(io);
    put(io,"    7         3      0.000000E+00");
    new_line(io);
    put(io,"    8         3      0.000000E+00");
    new_line(io);

    -- force
    new_line(io);
    new_line(io);
    put(io," node      axis             force");
    new_line(io);
    put(io,":");
    new_line(io);
    put(io,"   24");
    new_line(io);

    new_line(io);
    for glob in 1 .. 12 loop
      if xo(glob) /= 0.0 then
        put(io,glob,5);
        put(io,"         1     ");
        put(io,-xo(glob),2,6,3);
        new_line(io);
      end if;

      if yo(glob) /= 0.0 then
        put(io,glob,5);
        put(io,"         2     ");
        put(io,-yo(glob),2,6,3);
        new_line(io);
      end if;

      if zo(glob) /= 0.0 then
        put(io,glob,5);
        put(io,"         3     ");
        put(io,-zo(glob),2,6,3);
        new_line(io);
      end if;
    end loop;

    close(io);
  end input;

  procedure autocad is

    use
      text_io,
      ada.integer_text_io,
      ada.long_float_text_io;

    glob : integer;
    h    : long_float;
    io   : file_type;

  begin
    -- text height
    h := 0.075;

    create(io, out_file, "icosahedron.scr");

    -- set running object snap mode to NONE
    put(io,"osmode 0");
    new_line(io);

    -- icosahedron
    put(io,"layer make icosahedron ");
    new_line(io);

    for e in 1 .. 30 loop
      put(io,"3dpoly ");
      for loc in 1 .. 2 loop
        glob := edge(e,loc);
        put(io,xo(glob),1,8,3);
        put(io,",");
        put(io,yo(glob),1,8,3);
        put(io,",");
        put(io,zo(glob),1,8,3);
        put(io," ");
      end loop;
      new_line(io);
    end loop;

    -- icosahedron
    put(io,"layer make Vertices ");
    new_line(io);

    for glob in 1 .. 12 loop
      put(io,"ucs za ");
      put(io,xo(glob),1,8,3);
      put(io,",");
      put(io,yo(glob),1,8,3);
      put(io,",");
      put(io,zo(glob),1,8,3);
      put(io," ");
      put(io,2.0 * xo(glob),1,8,3);
      put(io,",");
      put(io,2.0 * yo(glob),1,8,3);
      put(io,",");
      put(io,2.0 * zo(glob),1,8,3);
      new_line(io);

      put(io,"text j m 0.0,0.0,0.0 ");
      put(io,h,1,8,3);
      put(io," 0 ");
      put(io,glob,0);
      new_line(io);

      put(io,"ucs w");
      new_line(io);
    end loop;

    -- autocad
    put(io,"layer set 0 ");
    new_line(io);

    put(io,"layer off Vertices ");
    new_line(io);

    put(io,"zoom ext ext");
    new_line(io);

    close(io);
  end autocad;

begin
  initia;
  input;
  autocad;
end icosahedron;
