-- NLP is a package for the unconstrained minimization of a
-- smooth nonlinear real-valued function F(X). Quasi-Newton
-- and line search with cubic interpolations are employed.
-- Descriptions of these methods can be found in the following
-- books:

-- Linear and Nonlinear Programming
-- David G. Luenberger
-- Addison-Wesley, 1984, ISBN 0-201-15794-2

-- Numerical Optimization
-- Jorge Nocedal, Stephen J. Wright
-- Springer-Verlag, 1999, ISBN 0-387-98793-2

-- N : The number of variables.

-- X : The vector of variables.

-- F : The smooth nonlinear real-valued function F(X).

-- G : The gradient vector of F(X).

-- OBJECTIVE : A procedure that evaluates function value and
-- gradient vector.

-- MINOR : The number of iterations required.

-- MAXMINOR : The maximum number of iterations allowed. It
-- must be greater that zero. It is usually 10 * N.

-- MAXNORM : The maximum value allowed for the infinity norm of
-- the gradient. The iterations terminate if the infinity norm
-- of the gradient becomes less than or equal to MAXNORM.

-- OPTIMAL : The state at finishing. It assumes true if
-- success, false otherwise.

-- MAXLS : The maximum number of cubic interpolations allowed.
-- It must be greater that one. It is usually 20.

-- LSTOL : Controls the accuracy of the search. It must lie in
-- the range 0.0 <= LSTOL < 1.0. Decreasing this value tends
-- to increase the accuracy of the search. It is usually 0.1.

-- MAXLIST : The number of BFGS corrections kept. These
-- corrections are stored separately on a doubly linked
-- circular list. Used only with the limited memory BFGS
-- method. It is usually between 3 and 20.

-- Vinicius F Arcaro
-- April, 2002

with
  terms;

package nlp is

  halt : exception;

  function norm (n : in integer;
                 v : in terms.fmat1_ptr) return long_float;

  generic
    with procedure objective (n : in integer;
                              f : out long_float;
                              x : in terms.fmat1_ptr;
                              g : in terms.fmat1_ptr);

  package qnewton is

    procedure lbfgs (maxlist,n,maxminor,maxls : in integer;
                     minor                    : out integer;
                     maxnorm,lstol            : in long_float;
                     f                        : out long_float;
                     x,g                      : in terms.fmat1_ptr;
                     optimal                  : out boolean);

    procedure mbfgs (n,maxminor,maxls : in integer;
                     minor            : out integer;
                     maxnorm,lstol    : in long_float;
                     f                : out long_float;
                     x,g              : in terms.fmat1_ptr;
                     optimal          : out boolean);

    procedure bfgs (n,maxminor,maxls : in integer;
                    minor            : out integer;
                    maxnorm,lstol    : in long_float;
                    f                : out long_float;
                    x,g              : in terms.fmat1_ptr;
                    optimal          : out boolean);

  end qnewton;
end nlp;
